#define       PAT_UNCACHEABLE         0x00
#define       PAT_WRITE_BACK          0x06
#define SD_F_SOFTPCMVOL           0x00000004
#define     PG_PTE_PAT      0x080
#define    PG_PDE_PAT      0x1000
#define       CPUID_PAT       0x00010000
#define       PAT_WRITE_THROUGH       0x04
#define       PAT_UNCACHED            0x07
#define       PAT_WRITE_COMBINING     0x01
#define       PAT_WRITE_PROTECTED     0x05

#include <i386/include/vmparam.h>
#include <sys/sched.h>
#include <i386/include/pmap.h>
#include <sys/smp.h>

#define pmap_pde(m, v)  (&((m)->pm_pdir[(vm_offset_t)(v) >> PDRSHIFT]))
#define  APIC_LOCAL_INTS 240
#define APIC_IPI_INTS (APIC_LOCAL_INTS + 2)
#define  IPI_INVLCACHE   (APIC_IPI_INTS + 4)

int     pmap_change_attr(vm_offset_t, vm_size_t, int);
void mix_setparentchild(struct snd_mixer *m, u_int32_t parent, u_int32_t childs);
void mix_setrealdev(struct snd_mixer *m, u_int32_t dev, u_int32_t realdev);
static int pmap_cache_bits(int mode, boolean_t is_pde);
void pmap_invalidate_cache(void);
static void smp_tlb_shootdown(u_int vector, vm_offset_t addr1, vm_offset_t addr2);
void smp_cache_flush(void);
void ipi_all_but_self(int ipi);

extern vm_offset_t smp_tlb_addr1;
extern vm_offset_t smp_tlb_addr2;
extern volatile int smp_tlb_wait;
extern    u_int   cpu_feature;

static void
smp_tlb_shootdown(u_int vector, vm_offset_t addr1, vm_offset_t addr2)
{
	u_int ncpu;

	ncpu = mp_ncpus - 1;	/* does not shootdown self */
	if (ncpu < 1)
		return;		/* no other cpus */
	mtx_assert(&smp_ipi_mtx, MA_OWNED);
	smp_tlb_addr1 = addr1;
	smp_tlb_addr2 = addr2;
	atomic_store_rel_int(&smp_tlb_wait, 0);
	ipi_all_but_self(vector);
	while (smp_tlb_wait < ncpu)
		ia32_pause();
}

void smp_cache_flush(void){
        if (smp_started)
                smp_tlb_shootdown(IPI_INVLCACHE, 0, 0);
}

void
pmap_invalidate_cache(void)
{

        sched_pin();
        wbinvd();
        smp_cache_flush();
        sched_unpin();
}

int
pmap_change_attr(va, size, mode)
	vm_offset_t va;
	vm_size_t size;
	int mode;
{
	vm_offset_t base, offset, tmpva;
	pt_entry_t *pte;
	u_int opte, npte;
	pd_entry_t *pde;

	base = va & PG_FRAME;
	offset = va & PAGE_MASK;
	size = roundup(offset + size, PAGE_SIZE);

	/* Only supported on kernel virtual addresses. */
	if (base <= VM_MAXUSER_ADDRESS)
		return (EINVAL);

	/* 4MB pages and pages that aren't mapped aren't supported. */
	for (tmpva = base; tmpva < (base + size); tmpva += PAGE_SIZE) {
		pde = pmap_pde(kernel_pmap, tmpva);
		if (*pde & PG_PS)
			return (EINVAL);
		if (*pde == 0)
			return (EINVAL);
		pte = vtopte(va);
		if (*pte == 0)
			return (EINVAL);
	}

	/*
	 * Ok, all the pages exist and are 4k, so run through them updating
	 * their cache mode.
	 */
	for (tmpva = base; size > 0; ) {
		pte = vtopte(tmpva);

		/*
		 * The cache mode bits are all in the low 32-bits of the
		 * PTE, so we can just spin on updating the low 32-bits.
		 */
		do {
			opte = *(u_int *)pte;
			npte = opte & ~(PG_PTE_PAT | PG_NC_PCD | PG_NC_PWT);
			npte |= pmap_cache_bits(mode, 0);
		} while (npte != opte &&
		    !atomic_cmpset_int((u_int *)pte, opte, npte));
		tmpva += PAGE_SIZE;
		size -= PAGE_SIZE;
	}

	/*
	 * Flush CPU caches to make sure any data isn't cached that shouldn't
	 * be, etc.
	 */    
	pmap_invalidate_range(kernel_pmap, base, tmpva);
	pmap_invalidate_cache();
	return (0);
}

#define MIXER_NAMELEN	16
struct snd_mixer {
	KOBJ_FIELDS;
	const char *type;
	void *devinfo;
	int busy;
	int hwvol_muted;
	int hwvol_mixer;
	int hwvol_step;
	device_t dev;
	u_int32_t hwvol_mute_level;
	u_int32_t devs;
	u_int32_t recdevs;
	u_int32_t recsrc;
	u_int16_t level[32];
	u_int8_t parent[32];
	u_int32_t child[32];
	u_int8_t realdev[32];
	char name[MIXER_NAMELEN];
	struct mtx *lock;
};

void
mix_setparentchild(struct snd_mixer *m, u_int32_t parent, u_int32_t childs)
{
	u_int32_t mask = 0;
	int i;

	if (m == NULL || parent >= SOUND_MIXER_NRDEVICES)
		return;
	for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
		if (i == parent)
			continue;
		if (childs & (1 << i)) {
			mask |= 1 << i;
			//if (m->parent[i] < SOUND_MIXER_NRDEVICES);
				m->child[m->parent[i]] &= ~(1 << i);
			m->parent[i] = parent;
			m->child[i] = 0;
		}
	}
	mask &= ~(1 << parent);
	m->child[parent] = mask;
}

void
mix_setrealdev(struct snd_mixer *m, u_int32_t dev, u_int32_t realdev)
{
	if (m == NULL || dev >= SOUND_MIXER_NRDEVICES ||
	    !(realdev == SOUND_MIXER_NONE || realdev < SOUND_MIXER_NRDEVICES))
		return;
	m->realdev[dev] = realdev;
}

static int
pmap_cache_bits(int mode, boolean_t is_pde)
{
	int pat_flag, pat_index, cache_bits;

	/* The PAT bit is different for PTE's and PDE's. */
	pat_flag = is_pde ? PG_PDE_PAT : PG_PTE_PAT;

	/* If we don't support PAT, map extended modes to older ones. */
	if (!(cpu_feature & CPUID_PAT)) {
		switch (mode) {
		case PAT_UNCACHEABLE:
		case PAT_WRITE_THROUGH:
		case PAT_WRITE_BACK:
			break;
		case PAT_UNCACHED:
		case PAT_WRITE_COMBINING:
		case PAT_WRITE_PROTECTED:
			mode = PAT_UNCACHEABLE;
			break;
		}
	}
	
	/* Map the caching mode to a PAT index. */
	switch (mode) {
#ifdef PAT_WORKS
	case PAT_UNCACHEABLE:
		pat_index = 3;
		break;
	case PAT_WRITE_THROUGH:
		pat_index = 1;
		break;
	case PAT_WRITE_BACK:
		pat_index = 0;
		break;
	case PAT_UNCACHED:
		pat_index = 2;
		break;
	case PAT_WRITE_COMBINING:
		pat_index = 5;
		break;
	case PAT_WRITE_PROTECTED:
		pat_index = 4;
		break;
#else
	case PAT_UNCACHED:
	case PAT_UNCACHEABLE:
	case PAT_WRITE_PROTECTED:
		pat_index = 3;
		break;
	case PAT_WRITE_THROUGH:
		pat_index = 1;
		break;
	case PAT_WRITE_BACK:
		pat_index = 0;
		break;
	case PAT_WRITE_COMBINING:
		pat_index = 2;
		break;
#endif
	default:
		panic("Unknown caching mode %d\n", mode);
	}	

	/* Map the 3-bit index value into the PAT, PCD, and PWT bits. */
	cache_bits = 0;
	if (pat_index & 0x4)
		cache_bits |= pat_flag;
	if (pat_index & 0x2)
		cache_bits |= PG_NC_PCD;
	if (pat_index & 0x1)
		cache_bits |= PG_NC_PWT;
	return (cache_bits);
}
